/* Example of correlation between columns in different tables. Multi-column 
histograms on a table can't fix.  Run multiple times to see if cardinality feedback
kicks in.
*/

set linesize 1000
set pagesize 1000
set echo on
set termout on

spool multitablecardinality2.log

/* DIVISION table with just two rows which point to SALES table.
Idea is that where the DIVNAME column='Mayberry' there are few records in the SALES table.
Where the DIVNAME column='Los Angeles' there are many records.  But the optimizer can't figure this out
given the statistics it has available. */

drop table DIVISION;
drop table SALES;

create table DIVISION (DIVNAME VARCHAR2(2000),DIVNUM NUMBER);
create table SALES (DIVNUM NUMBER);

insert into DIVISION values ('Los Angeles',1);
insert into DIVISION values ('Mayberry',2);

commit;

/* SALES table has 1,000,000 1's for LA and one 2 for Mayberry */

declare
  i number;
begin
  i:=1;
  loop
    insert into SALES values (1);
    i:=i+1;
    if (mod(i,10000)=0) then
      commit;
    end if;
    exit when (i > 1000000);
  end loop;
  commit;
  insert into SALES values (2);
  commit;
end;
/

create index SALESINDEX on SALES(DIVNUM); 

/* Gather compute stats with max number of histograms all columns and multi column stats on DIVISION table's columns */


select DBMS_STATS.CREATE_EXTENDED_STATS (NULL,'DIVISION','(DIVNAME,DIVNUM)') from dual;
execute DBMS_STATS.GATHER_TABLE_STATS (NULL,'DIVISION',cascade=>TRUE,estimate_percent=>NULL,method_opt=>'FOR ALL COLUMNS SIZE 254');
execute DBMS_STATS.GATHER_TABLE_STATS (NULL,'SALES',cascade=>TRUE,estimate_percent=>NULL,method_opt=>'FOR ALL COLUMNS SIZE 254');

/* Show how long it takes with the wrong plan. */

set timing on

select B.DIVNUM 
from DIVISION A,SALES B
where
a.DIVNUM=B.DIVNUM and
A.DIVNAME='Mayberry';

set timing off

select * from table(dbms_xplan.display_cursor(null,null,'ALL'));

set timing on

/* Show how long it takes with the index being used correctly.  */

select /*+ INDEX(B SALESINDEX) NO_INDEX_FFS(B SALESINDEX)  */ B.DIVNUM 
from DIVISION A,SALES B
where
a.DIVNUM=B.DIVNUM and
A.DIVNAME='Mayberry';

set timing off

select * from table(dbms_xplan.display_cursor(null,null,'ALL'));

-- column info

select column_name,
num_distinct,num_buckets,
to_char(last_analyzed,'YYYY-MM-DD HH24:MI:SS') LA
from USER_TAB_COLS 
where table_name='DIVISION';

-- run some more times to see if plan changes due to
-- cardinality feedback

set timing on

select B.DIVNUM 
from DIVISION A,SALES B
where
a.DIVNUM=B.DIVNUM and
A.DIVNAME='Mayberry';

/
/
/
/
/

set timing off

select * from table(dbms_xplan.display_cursor(null,null,'ALL'));

-- try dynamic sampling - I've seen this help with single table
-- correlated predicates.

set timing on

select /*+ DYNAMIC_SAMPLING(A 10) DYNAMIC_SAMPLING(B 10)*/ 
B.DIVNUM 
from DIVISION A,SALES B
where
a.DIVNUM=B.DIVNUM and
A.DIVNAME='Mayberry';

set timing off

select * from table(dbms_xplan.display_cursor(null,null,'ALL'));

spool off
